<?php

declare(strict_types=1);

namespace XPInvest\Reporting\DI;

use Doctrine\DBAL\Configuration;
use Doctrine\DBAL\Logging\LoggerChain;
use Fmasa\SentryBreadcrumbsMonologHandler\BreadcrumbsHandler;
use Monolog\Formatter\LogstashFormatter;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Nette\DI\CompilerExtension;
use Nette\DI\Config\Helpers;
use Nette\DI\Definitions\ServiceDefinition;
use Nette\DI\Definitions\Statement;
use Sentry\Monolog\Handler;
use Sentry\SentrySdk;
use Tracy\Bridges\Psr\PsrToTracyLoggerAdapter;
use XPInvest\Reporting\DBAL\SQLLogger;
use XPInvest\Reporting\Sentry\HubFactory;

final class ReportingExtension extends CompilerExtension
{
    private const DEFAULTS = [
        'sentry' => [
            'dsn' => null,
            'eventProcessors' => [],
            'release' => null,
        ],
    ];

    public function loadConfiguration() : void
    {
        $this->registerMonologLogger();
        $this->registerSentry();
    }

    public function beforeCompile() : void
    {
        $builder = $this->getContainerBuilder();

        $loggerService = $builder->getDefinitionByType(Logger::class);
        assert($loggerService instanceof ServiceDefinition);

        $loggerService->addSetup('setHandlers', [$builder->findByType(HandlerInterface::class)]);

        $this->registerSQLLogger();
        $this->replaceDefaultTracyLoggerByPsrBridge();
    }

    private function replaceDefaultTracyLoggerByPsrBridge() : void
    {
        $builder = $this->getContainerBuilder();
        $logger = $builder->getDefinition('tracy.logger');

        $this->getInitialization()
            ->addBody($builder->formatPhp('Tracy\Debugger::setLogger(?);', [$logger]));
        $builder->getDefinition($this->prefix('logger'));
    }

    private function registerMonologLogger() : void
    {
        $builder = $this->getContainerBuilder();
        $applicationName = $this->config()['name'];

        $builder->addDefinition($this->prefix('logger'))
            ->setFactory(Logger::class, [$applicationName]);

        $formatter = new Statement(LogstashFormatter::class, ['applicationName' => $applicationName]);

        $builder->addDefinition($this->prefix('logger.stdoutHandler'))
            ->setFactory(StreamHandler::class, ['php://stdout'])
            ->addSetup('setFormatter', [$formatter]);

        $builder->removeDefinition('tracy.logger');
        $builder->addDefinition('tracy.logger')
            ->setFactory(PsrToTracyLoggerAdapter::class);
    }

    private function registerSentry() : void
    {
        $builder = $this->getContainerBuilder();
        $config = $this->config()['sentry'];

        $eventProcessors = [];

        foreach ($config['eventProcessors'] as $index => $eventProcessor) {
            assert($eventProcessor instanceof Statement);

            $eventProcessors[] = $builder->addDefinition($this->prefix('sentry.eventProcessor.' . $index))
                ->setFactory($eventProcessor);
        }

        $builder->addDefinition($this->prefix('sentry.hub'))
            ->setFactory(HubFactory::class . '::create', [$config['dsn'], $config['release'], $eventProcessors])
            ->addSetup(SentrySdk::class . '::setCurrentHub');

        $builder->addDefinition($this->prefix('sentry.handler'))
            ->setFactory(Handler::class, ['level' => Logger::ERROR]);

        $builder->addDefinition($this->prefix('sentry.breadcrumbsHandler'))
            ->setFactory(BreadcrumbsHandler::class);
    }

    private function registerSQLLogger() : void
    {
        $builder = $this->getContainerBuilder();

        foreach ($builder->findByType(Configuration::class) as $configurationService) {
            assert($configurationService instanceof ServiceDefinition);

            $configurationService->addSetup('setSQLLogger', [new Statement(LoggerChain::class)]);
            $configurationService->addSetup('$service->getSQLLogger()->addLogger(?)', [new Statement(SQLLogger::class)]);
        }
    }

    /**
     * @return array<string, mixed>
     */
    private function config() : array
    {
        return Helpers::merge($this->getConfig(), self::DEFAULTS);
    }
}
