<?php

declare(strict_types=1);

namespace XPInvest\Reporting\DI;

use Doctrine\DBAL\Configuration;
use Monolog\Formatter\LogstashFormatter;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\StreamHandler;
use Monolog\Level;
use Monolog\Logger;
use Nette\DI\CompilerExtension;
use Nette\DI\Definitions\ServiceDefinition;
use Nette\DI\Definitions\Statement;
use Nette\Schema\Helpers;
use Sentry\Monolog\Handler;
use Sentry\SentrySdk;
use Tracy\Bridges\Psr\PsrToTracyLoggerAdapter;
use XPInvest\Reporting\DBAL\SQLLogger;
use XPInvest\Reporting\Monolog\BreadcrumbsHandler;
use XPInvest\Reporting\Sentry\HubFactory;

final class ReportingExtension extends CompilerExtension
{
    private const array DEFAULTS = [
        'sentry' => [
            'dsn' => null,
            'eventProcessors' => [],
            'release' => null,
        ],
    ];

    public function loadConfiguration() : void
    {
        $this->registerMonologLogger();
        $this->registerSentry();
    }

    public function beforeCompile() : void
    {
        $builder = $this->getContainerBuilder();

        $loggerService = $builder->getDefinitionByType(Logger::class);
        assert($loggerService instanceof ServiceDefinition);

        $loggerService->addSetup('setHandlers', [$builder->findByType(HandlerInterface::class)]);

        $this->registerSQLLogger();
        $this->replaceDefaultTracyLoggerByPsrBridge();
    }

    private function replaceDefaultTracyLoggerByPsrBridge() : void
    {
        $builder = $this->getContainerBuilder();

        $this->getInitialization()
            ->addBody('Tracy\Debugger::setLogger($this->getService(?));', ['tracy.logger']);
        $builder->getDefinition($this->prefix('logger'));
    }

    private function registerMonologLogger() : void
    {
        $builder = $this->getContainerBuilder();
        $applicationName = $this->config()['name'];

        $builder->addDefinition($this->prefix('logger'))
            ->setFactory(Logger::class, [$applicationName]);

        $formatter = new Statement(LogstashFormatter::class, ['applicationName' => $applicationName]);

        $builder->addDefinition($this->prefix('logger.stdoutHandler'))
            ->setFactory(StreamHandler::class, ['php://stdout', Level::Info->value])
            ->addSetup('setFormatter', [$formatter]);

        $builder->removeDefinition('tracy.logger');
        $builder->addDefinition('tracy.logger')
            ->setFactory(PsrToTracyLoggerAdapter::class);
    }

    private function registerSentry() : void
    {
        $builder = $this->getContainerBuilder();
        $config = $this->config()['sentry'];

        $eventProcessors = [];

        foreach ($config['eventProcessors'] as $index => $eventProcessor) {
            assert($eventProcessor instanceof Statement);

            $eventProcessors[] = $builder->addDefinition($this->prefix('sentry.eventProcessor.' . $index))
                ->setFactory($eventProcessor);
        }

        $builder->addDefinition($this->prefix('sentry.hub'))
            ->setFactory(HubFactory::class . '::create', [$config['dsn'], $config['release'], $eventProcessors])
            ->addSetup(SentrySdk::class . '::setCurrentHub');

        $builder->addDefinition($this->prefix('sentry.handler'))
            ->setFactory(Handler::class, ['level' => Level::Error->value]);

        $builder->addDefinition($this->prefix('sentry.breadcrumbsHandler'))
            ->setFactory(BreadcrumbsHandler::class);
    }

    private function registerSQLLogger() : void
    {
        $builder = $this->getContainerBuilder();

        $serviceName = $builder->getByType(Configuration::class);

        if ($serviceName === null) {
            return;
        }

        $configurationService = $builder->getDefinition($serviceName);
        assert($configurationService instanceof ServiceDefinition);

        $configurationService->addSetup('setSQLLogger', [new Statement(SQLLogger::class)]);
    }

    /**
     * @return array<string, mixed>
     */
    private function config() : array
    {
        return Helpers::merge($this->getConfig(), self::DEFAULTS);
    }
}
