<?php

declare(strict_types=1);

namespace XPInvest\Reporting\DI;

use Doctrine\DBAL\Configuration;
use Fmasa\SentryBreadcrumbsMonologHandler\BreadcrumbsHandler;
use Monolog\Formatter\LogstashFormatter;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;
use Nette\DI\CompilerExtension;
use Nette\DI\Config\Helpers;
use Nette\DI\ServiceDefinition;
use Nette\DI\Statement;
use Sentry\Monolog\Handler;
use Sentry\State\Hub;
use Tracy\Bridges\Psr\PsrToTracyLoggerAdapter;
use XPInvest\Reporting\DBAL\SQLLogger;
use XPInvest\Reporting\Sentry\HubFactory;

final class ReportingExtension extends CompilerExtension
{
    private const LOG_RETENTION_IN_DAYS = 'logRetentionInDays';
    private const LOG_FILE = 'logFile';

    private const DEFAULTS = [
        self::LOG_RETENTION_IN_DAYS => 7,
        'sentry' => [
            'dsn' => null,
            'eventProcessors' => [],
        ],
    ];

    public function loadConfiguration() : void
    {
        $this->registerMonologLogger();
        $this->registerSentry();
    }

    public function beforeCompile() : void
    {
        $builder = $this->getContainerBuilder();

        $loggerService = $builder->getDefinitionByType(Logger::class);
        assert($loggerService instanceof ServiceDefinition);

        $loggerService->addSetup('setHandlers', [$builder->findByType(HandlerInterface::class)]);

        $this->registerSQLLogger();
    }

    private function registerMonologLogger() : void
    {
        $builder = $this->getContainerBuilder();
        $applicationName = $this->config()['name'];
        $logFile = $this->config()[self::LOG_FILE];

        $logDirectory = dirname($logFile);

        if (! is_dir($logDirectory)) {
            throw new \RuntimeException(sprintf('Log directory "%s" does not exist', $logDirectory));
        }

        $builder->addDefinition($this->prefix('logger'))
            ->setFactory(Logger::class, [$applicationName]);

        $formatter = new Statement(LogstashFormatter::class, [
            'applicationName' => $applicationName,
            'version' => LogstashFormatter::V1
        ]);

        $builder->addDefinition($this->prefix('logger.fileHandler'))
            ->setFactory(
                RotatingFileHandler::class,
                [
                    $this->config()[self::LOG_FILE],
                    $this->config()[self::LOG_RETENTION_IN_DAYS],
                    Logger::INFO,
                ]
            )->addSetup('setFormatter', [$formatter]);

        $builder->removeDefinition('tracy.logger');
        $builder->addDefinition('tracy.logger')
            ->setFactory(PsrToTracyLoggerAdapter::class);
    }

    private function registerSentry() : void
    {
        $builder = $this->getContainerBuilder();

        $eventProcessors = [];

        foreach ($this->config()['sentry']['eventProcessors'] as $index => $eventProcessor) {
            assert($eventProcessor instanceof Statement);

            $eventProcessors[] = $builder->addDefinition($this->prefix('sentry.eventProcessor.' . $index))
                ->setFactory($eventProcessor);
        }

        $builder->addDefinition($this->prefix('sentry.hub'))
            ->setFactory(HubFactory::class . '::create', [$this->config()['sentry']['dsn'], $eventProcessors])
            ->addSetup(Hub::class . '::setCurrent');

        $builder->addDefinition($this->prefix('sentry.handler'))
            ->setFactory(Handler::class, ['level' => Logger::ERROR]);

        $builder->addDefinition($this->prefix('sentry.breadcrumbsHandler'))
            ->setFactory(BreadcrumbsHandler::class);
    }

    private function registerSQLLogger() : void
    {
        $builder = $this->getContainerBuilder();

        $serviceName = $builder->getByType(Configuration::class);

        if ($serviceName === null) {
            return;
        }

        $configurationService = $builder->getDefinition($serviceName);
        assert($configurationService instanceof ServiceDefinition);

        $configurationService->addSetup('setSQLLogger', [new Statement(SQLLogger::class)]);
    }

    /**
     * @return array<string, mixed>
     */
    private function config() : array
    {
        return Helpers::merge($this->getConfig(), self::DEFAULTS);
    }
}
