<?php

declare(strict_types=1);

namespace XPInvest\Reporting\DI;

use Doctrine\DBAL\Configuration;
use Doctrine\DBAL\Logging\LoggerChain;
use Fmasa\SentryBreadcrumbsMonologHandler\BreadcrumbsHandler;
use Monolog\Handler\NullHandler;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\TestHandler;
use Monolog\Logger;
use Nette\Configurator;
use Nette\DI\Container;
use PHPUnit\Framework\TestCase;
use RuntimeException;
use Sentry\Integration\ModulesIntegration;
use Sentry\Integration\RequestIntegration;
use Sentry\Monolog\Handler;
use Sentry\SentrySdk;
use Sentry\State\Hub;
use Sentry\State\HubInterface;
use Tracy\Bridges\Psr\PsrToTracyLoggerAdapter;
use Tracy\Debugger;

final class ReportingExtensionTest extends TestCase
{
    public function testSentryIsRegisteredWithDSN() : void
    {
        $container = $this->createContainer(__DIR__ . '/withSentryDsn.neon');

        $hub = $container->getByType(Hub::class);

        $this->assertInstanceOf(Hub::class, $hub);
        $this->assertSame('https://foo@sentry.io/123', (string) $hub->getClient()->getOptions()->getDsn());
        $this->assertTrue($hub->getClient()->getOptions()->shouldAttachStacktrace());
    }

    public function testSentryDSNIsNullByDefault() : void
    {
        $container = $this->createContainer(__DIR__ . '/additionalHandler.neon');
        $this->assertNull($container->getByType(Hub::class)->getClient()->getOptions()->getDsn());
    }

    public function testSentryMonologHandlersAreRegistered() : void
    {
        $container = $this->createContainer(__DIR__ . '/withSentryDsn.neon');

        $handlers = $container->getByType(Logger::class)->getHandlers();

        $this->assertCount(3, $handlers);
        $this->assertInstanceOf(RotatingFileHandler::class, $handlers[0]);
        $this->assertInstanceOf(Handler::class, $handlers[1]);
        $this->assertInstanceOf(BreadcrumbsHandler::class, $handlers[2]);
    }

    public function testIfLogFileDoesNotPointToValidDirectoryExceptionIsThrown() : void
    {
        $this->expectException(RuntimeException::class);
        $this->expectExceptionMessage('Log directory "foo" does not exist');
        $this->createContainer(__DIR__ . '/invalidLogDirectory.neon');
    }

    public function testTracyLoggerIsReplacedWithPsrAdapter() : void
    {
        $container = $this->createContainer(__DIR__ . '/withSentryDsn.neon');

        $this->assertInstanceOf(PsrToTracyLoggerAdapter::class, $container->getService('tracy.logger'));
        $this->assertInstanceOf(PsrToTracyLoggerAdapter::class, Debugger::getLogger());
    }

    public function testRegisterAdditionalHandler() : void
    {
        $container = $this->createContainer(__DIR__ . '/additionalHandler.neon');

        $handlers = $container->getByType(Logger::class)->getHandlers();

        $this->assertCount(4, $handlers);
        $this->assertInstanceOf(NullHandler::class, $handlers[3]);
    }

    public function testSQLLoggerIsRegisteredIfDBALConfigurationExists() : void
    {
        $container = $this->createContainer(__DIR__ . '/sqlLogger.neon');
        $sqlLogger = $container->getService($container->findByType(Configuration::class)[0])->getSQLLogger();

        $testHandler = new TestHandler();
        $container->getByType(Logger::class)->pushHandler($testHandler);
        $sqlLogger->startQuery('SELECT * FROM users');
        $sqlLogger->stopQuery();

        $this->assertInstanceOf(LoggerChain::class, $sqlLogger);
        $this->assertCount(1, $testHandler->getRecords());
    }

    public function testSentryHubIsRegisteredAsCurrent() : void
    {
        $container = $this->createContainer(__DIR__ . '/withSentryDsn.neon');
        $this->assertSame($container->getByType(HubInterface::class), SentrySdk::getCurrentHub());
    }

    public function testSentryIntegrationsAreRegistered() : void
    {
        $container = $this->createContainer(__DIR__ . '/withSentryDsn.neon');
        $hub = $container->getByType(HubInterface::class);

        foreach ([RequestIntegration::class, ModulesIntegration::class] as $integration) {
            $this->assertInstanceOf($integration, $hub->getIntegration($integration));
        }
    }

    public function testReleaseIsSetInSentry() : void
    {
        $container = $this->createContainer(__DIR__ . '/withSentryRelease.neon');
        $hub = $container->getByType(HubInterface::class);

        $this->assertSame(
            'frontend/project@123',
            $hub->getClient()->getOptions()->getRelease()
        );
    }

    private function createContainer(string $configFile) : Container
    {
        $configurator = new Configurator();
        $configurator->addConfig($configFile);
        $configurator->setDebugMode(true);
        $configurator->setTempDirectory(__DIR__ . '/temp');

        return $configurator->createContainer();
    }
}
