<?php declare(strict_types=1);

namespace XPInvest\PayUButton\DI;

use Nette\DI\CompilerExtension;
use Nette\DI\Config\Helpers;
use Nette\DI\Statement;
use Nette\Http\SessionSection;
use XPInvest\PayUButton\Client;
use XPInvest\PayUButton\UI\IButtonFactory;

class PayUButtonExtension extends CompilerExtension
{

    /** @var array */
    private $defaults = [
        'sandbox' => true,
        'clientId' => null,
        'secret' => null,
        'key' => null,
        'cache' => null,
        'autowire' => false,
    ];

    public function loadConfiguration()
    {
        $builder = $this->getContainerBuilder();

        $mainConfig = $this->getConfig();

        foreach ($mainConfig['instances'] as $name => $instance) {
            if (!is_string($name)) {
                throw new \InvalidArgumentException('Instance name must be string');
            }

            $config = Helpers::merge($instance, $this->defaults);

            if (isset($config['cache']) && !$config['cache'] instanceof \OauthCacheInterface) {
                throw new \InvalidArgumentException('Cache must be instance of \OauthCacheInterface.');
            }

            if ($config['cache'] === null) {
                $directory = $this->getCacheDirectory($name);
                $cache = new Statement(\OauthCacheFile::class, ['directory' => $directory]);
            } else {
                $cache = $config['cache'];
            }

            $clientName = $this->prefix($name . '.apiClient');

            $builder->addDefinition($clientName)
                ->setFactory(Client::class)
                ->setAutowired($config['autowire'])
                ->setArguments([
                    $mainConfig['sandbox'],
                    $config['clientId'],
                    $config['secret'],
                    $config['key'],
                    $cache
                ]);

            $builder->addDefinition($this->prefix($name . '.buttonFactory'))
                ->setImplement(IButtonFactory::class)
                ->setAutowired($config['autowire'])
                ->setArguments([
                    'session' => new Statement(SessionSection::class, ['name' => $config['session'] ?? "payu.$name"]),
                    'client' => '@' . $clientName
                ]);
        }
    }

    private function getCacheDirectory(string $instanceName): string
    {
        $directory = $this->getContainerBuilder()->parameters['tempDir'] . '/cache/PayU.Signature/' . $instanceName;

        $this->createDirectory($directory);

        return $directory;
    }

    private function createDirectory(string $directory)
    {
        if (!is_dir($directory)) {
            mkdir($directory, 0777, true);
        }
    }
}
