<?php declare(strict_types=1);

namespace XPInvest\PayUButton\UI;

use Nette\Application\UI\Control;
use Nette\Http\IRequest;
use Nette\Http\SessionSection;
use XPInvest\PayUButton\Client;
use XPInvest\PayUButton\Exceptions\InvalidArgumentException;
use XPInvest\PayUButton\Exceptions\OrderNotFoundException;
use XPInvest\PayUButton\Exceptions\PayUApiException;
use XPInvest\PayUButton\Exceptions\PayUException;
use XPInvest\PayUButton\Order;

class Button extends Control
{

	/** @var callable[] */
	public $onBeforeStart = [];

	/** @var callable[] */
	public $onStart = [];

	/** @var callable[] */
	public $onSuccess = [];

	/** @var callable[] */
	public $onError = [];

	/** @var NULL|Order */
	private $order;

	/** @var string */
	private $notifyUrl;

	/** @var IRequest */
	private $request;

	/** @var SessionSection */
	private $session;

	/** @var Client */
	private $client;

	/** @var bool */
	private $renderOutput = TRUE;

	/**
	 * Button constructor.
	 * @param IRequest $request
	 * @param SessionSection $session
	 * @param Client $client
	 * @param string $notifyUrl
	 * @param NULL|Order $order
	 */
	public function __construct(
		IRequest $request,
		SessionSection $session,
		Client $client,
		string $notifyUrl,
		Order $order = NULL)
	{
		parent::__construct();
		$this->request = $request;
		$this->session = $session;
		$this->client = $client;
		$this->notifyUrl = $notifyUrl;
		$this->order = $order;
	}

	public function render()
	{
		if(!$this->renderOutput) {
			return;
		}

		if(!$this->template->getFile()) {
			$this->template->setFile(__DIR__ . '/templates/Button.latte');
		}
		$this->template->render();
	}

	public function handlePay()
	{
		try {
			$this->onBeforeStart($this);
			$this->validate();
			$response = $this->client->createOrder(
				$this->order,
				$this->link('//complete!', ['id' => $this->order->getId()]),
				$this->notifyUrl);
			$this->session[$this->order->getId()] = $response->getOrderId();
			$this->onStart($response, $this->order);
			$this->presenter->redirectUrl($response->getRedirectUrl());
		} catch (PayUApiException $e) {
			$this->onError($e);
		}
	}

	public function handleComplete(string $id)
	{
		$this->renderOutput = FALSE;

		if(!isset($this->session[$id])) {
			$this->onError(new OrderNotFoundException);
			return;
		}

		try {
			$payment = $this->client->retrieveOrderPayment($this->session[$id]);
			$this->onSuccess($payment);
		} catch(PayUException $e) {
			$this->onError($e);
		}
	}

	/**
	 * @param Order $order
	 * @throws InvalidArgumentException
	 */
	public function setOrder(Order $order)
	{
		if($this->order) {
			throw new InvalidArgumentException('Order is already exists!');
		}

		$this->order = $order;
	}

	/**
	 * @throws InvalidArgumentException
	 */
	private function validate()
	{
		if(!$this->order) {
			throw new InvalidArgumentException('Order is missing!');
		}
	}

}
