<?php declare(strict_types=1);

namespace XPInvest\PayUButton;

use Money\Currency;
use Money\Money;
use Nette\Http\IRequest;
use XPInvest\PayUButton\Exceptions\InvalidRequestException;
use XPInvest\PayUButton\Exceptions\PayUApiException;
use XPInvest\PayUButton\Factory\NewOrderArrayFactory;
use XPInvest\PayUButton\Response\NewOrderResponse;

class Client
{

	/** @var bool */
	private $sandbox;

	/** @var string */
	private $clientId;

	/**
	 * Client constructor.
	 * @param bool $sandbox
	 * @param string $clientId
	 * @param string $secret
	 * @param string $key
	 * @param \OauthCacheInterface $oauthCache
	 * @throws \OpenPayU_Exception_Configuration
	 */
	public function __construct(
		bool $sandbox,
		string $clientId,
		string $secret,
		string $key,
		\OauthCacheInterface $oauthCache)
	{
		$this->sandbox = $sandbox;
		$this->clientId = $clientId;

		if($this->sandbox) {
			\OpenPayU_Configuration::setEnvironment('sandbox');
		}

		\OpenPayU_Configuration::setMerchantPosId($this->clientId);
		\OpenPayU_Configuration::setOauthClientId($this->clientId);
		\OpenPayU_Configuration::setOauthClientSecret($secret);
		\OpenPayU_Configuration::setSignatureKey($key);
		\OpenPayU_Configuration::setOauthTokenCache($oauthCache);
	}

	/**
	 * @param Order $order
	 * @param string $continueUrl
	 * @param string $notifyUrl
	 * @return NewOrderResponse
	 * @throws PayUApiException
	 */
	public function createOrder(Order $order, string $continueUrl, string $notifyUrl): NewOrderResponse
	{
		$data = NewOrderArrayFactory::create($order, $this->clientId, $continueUrl, $notifyUrl);
		try {
			$response = \OpenPayU_Order::create($data);
			if($response->getStatus() == \OpenPayU_Order::SUCCESS) {
				return new NewOrderResponse(
					$order->getId(),
					$response->getResponse()->orderId,
					$response->getResponse()->redirectUri);
			} else {
				throw new PayUApiException('Something went wrong during api communication.');
			}
		} catch (\OpenPayU_Exception $e) {
			throw new PayUApiException('Something went wrong during api communication.');
		}
	}

	/**
	 * @param string $id
	 * @return Order\Payment
	 * @throws PayUApiException
	 */
	public function retrieveOrderPayment(string $id): Order\Payment
	{
		try {
			$response = \OpenPayU_Order::retrieve($id);
			if($response->getStatus() == \OpenPayU_Order::SUCCESS) {
				$order = $response->getResponse()->orders[0];
				return new Order\Payment(
					$order->extOrderId,
					$order->orderId,
					new Money($order->totalAmount, new Currency($order->currencyCode)),
					$order->status
				);
			} else {
				throw new PayUApiException('Something went wrong during api communication.');
			}
		} catch(\OpenPayU_Exception $e) {
			throw new PayUApiException('Something went wrong during api communication.');
		}
	}

	/**
	 * @param IRequest $request
	 * @return Order\Payment
	 * @throws PayUApiException
	 */
	public function consumeOrderPaymentNotificationRequest(IRequest $request): Order\Payment
	{
		if(!$request->isMethod(IRequest::POST) || empty($request->getRawBody())) {
			throw new InvalidRequestException;
		}

		try {
			$response = \OpenPayU_Order::consumeNotification($request->getRawBody());
			$order = $response->getResponse()->order;
			return new Order\Payment(
				$order->extOrderId,
				$order->orderId,
				new Money($order->totalAmount, new Currency($order->currencyCode)),
				$order->status
			);
		} catch(\OpenPayU_Exception $e) {
			throw new PayUApiException('Something went wrong during api communication.');
		}
	}

	public function sendSuccessResponse()
	{
		header('HTTP/1.1 200 OK');
		exit;
	}

	public function sendErrorResponse()
	{
		header('HTTP/1.1 422 Error');
		exit;
	}

}
