<?php declare(strict_types=1);

namespace XPInvest\PayUButton\DI;

use Nette\DI\CompilerExtension;
use Nette\DI\Config\Helpers;
use Nette\DI\Statement;
use Nette\Http\SessionSection;
use XPInvest\PayUButton\Client;
use XPInvest\PayUButton\UI\IButtonFactory;

class PayUButtonExtension extends CompilerExtension
{

	/** @var array */
	private $defaults = [
		'sandbox' => TRUE,
		'clientId' => NULL,
		'secret' => NULL,
		'key' => NULL,
		'session' => 'payu',
		'cache' => NULL
	];

	public function loadConfiguration()
	{
		$config = Helpers::merge($this->getConfig(), $this->defaults);

		$builder = $this->getContainerBuilder();

		if($config['cache'] && !$config['cache'] instanceof \OauthCacheInterface) {
			throw new \InvalidArgumentException('Cache must be instance of \OauthCacheInterface.');
		} elseif(is_null($config['cache'])) {
			$directory = $this->getCacheDirectory();
			$cache = new Statement(\OauthCacheFile::class, ['directory' => $directory]);
		} else {
			$cache = $config['cache'];
		}

		$builder->addDefinition($this->prefix('apiClient'))
			->setFactory(Client::class)
			->setArguments([$config['sandbox'], $config['clientId'], $config['secret'], $config['key'], $cache]);

		$builder->addDefinition($this->prefix('buttonFactory'))
			->setImplement(IButtonFactory::class)
			->setArguments(['session' => new Statement(SessionSection::class, ['name' => $config['session']])]);
	}

	/**
	 * @return string
	 */
	private function getCacheDirectory(): string
	{
		$directory = $this->getContainerBuilder()->parameters['tempDir'] .
			DIRECTORY_SEPARATOR . 'cache' .
			DIRECTORY_SEPARATOR . 'PayU.Signature';

		$this->createDirectory($directory);

		return $directory;
	}

	private function createDirectory(string $directory)
	{
		if(!is_dir($directory)) {
			mkdir($directory);
		}
	}

}
