<?php declare(strict_types=1);

namespace XPInvest\PayUButton\UI;

use Nette\Application\UI\Control;
use Nette\Http\IRequest;
use Nette\Http\SessionSection;
use XPInvest\PayUButton\Client;
use XPInvest\PayUButton\Exceptions\OrderNotFoundException;
use XPInvest\PayUButton\Exceptions\PayUException;
use XPInvest\PayUButton\Order;

class Button extends Control
{

	/** @var callable[] */
	public $onStart = [];

	/** @var callable[] */
	public $onSuccess = [];

	/** @var callable[] */
	public $onError = [];

	/** @var Order */
	private $order;

	/** @var string */
	private $notifyUrl;

	/** @var IRequest */
	private $request;

	/** @var SessionSection */
	private $session;

	/** @var Client */
	private $client;

	/** @var bool */
	private $renderOutput = TRUE;

	/**
	 * Button constructor.
	 * @param Order $order
	 * @param string $notifyUrl
	 * @param IRequest $request
	 * @param SessionSection $session
	 * @param Client $client
	 */
	public function __construct(
		Order $order,
		string $notifyUrl,
		IRequest $request,
		SessionSection $session,
		Client $client)
	{
		parent::__construct();
		$this->order = $order;
		$this->notifyUrl = $notifyUrl;
		$this->request = $request;
		$this->session = $session;
		$this->client = $client;
	}

	/**
	 * @return bool
	 */
	public function isSandboxActive(): bool
	{
		return $this->client->isSandboxActive();
	}

	public function render()
	{
		if(!$this->renderOutput) {
			return;
		}

		if(!$this->template->getFile()) {
			$this->template->setFile(__DIR__ . '/templates/Button.latte');
		}
		$this->template->render();
	}

	public function handlePay()
	{
		try {
			$response = $this->client->createOrder(
				$this->order,
				$this->link('//complete!', ['id' => $this->order->getId()]),
				$this->notifyUrl);
			$this->session[$this->order->getId()] = $response->getOrderId();
			$this->onStart($response, $this->order);
			$this->presenter->redirectUrl($response->getRedirectUrl());
		} catch (PayUException $e) {
			$this->onError($e);
		}
	}

	public function handleComplete(string $id)
	{
		$this->renderOutput = FALSE;

		if(!isset($this->session[$id])) {
			$this->onError(new OrderNotFoundException);
			return;
		}

		try {
			$payment = $this->client->retrieveOrderPayment($this->session[$id]);
			$this->onSuccess($payment);
		} catch(PayUException $e) {
			$this->onError($e);
		}
	}

}
