<?php declare(strict_types=1);

namespace XPInvest\PayUButton;

use Money\Money;
use Nette\SmartObject;
use XPInvest\PayUButton\Exceptions\InvalidArgumentException;
use XPInvest\PayUButton\Order\Customer;
use XPInvest\PayUButton\Order\Product;

/**
 * @property-read string $id
 * @property-read Product[] $products
 * @property-read string $description
 * @property-read string $ip
 * @property-read NULL|Customer $customer
 * @property-read Money $total
 */
class Order
{

	use SmartObject;

	/** @var string */
	private $id;

	/** @var Product[] */
	private $products;

	/** @var string */
	private $description;

	/** @var string */
	private $ip;

	/** @var NULL|Customer */
	private $customer;

	/**
	 * Order constructor.
	 * @param string $id
	 * @param Product[] $products
	 * @param string $description
	 * @param string $ip
	 * @param NULL|Customer $customer
	 * @throws InvalidArgumentException
	 */
	public function __construct(string $id, array $products, string $description, string $ip, Customer $customer = NULL)
	{
		$this->id = $id;
		if(!count($products)) {
			throw new InvalidArgumentException('At least one product must be provided.');
		}
		$this->products = $products;
		$this->description = $description;
		$this->ip = $ip;
		$this->customer = $customer;
	}

	/**
	 * @return string
	 */
	public function getId(): string
	{
		return $this->id;
	}

	/**
	 * @return Product[]
	 */
	public function getProducts(): array
	{
		return $this->products;
	}

	/**
	 * @return NULL|Customer
	 */
	public function getCustomer(): Customer
	{
		return $this->customer;
	}

	/**
	 * @return string
	 */
	public function getDescription(): string
	{
		return $this->description;
	}

	/**
	 * @return string
	 */
	public function getIp(): string
	{
		return $this->ip;
	}

	/**
	 * @return Money
	 */
	public function getTotal(): Money
	{
		/** @var $product Product */
		$product = array_values($this->products)[0];

		$total = new Money(0, $product->getUnitPrice()->getCurrency());

		foreach($this->products as $product) {
			$total = $total->add($product->getTotal());
		}

		return $total;
	}

}
