<?php declare(strict_types=1);

namespace XPInvest\PayUButton\UI;

use Nette\Application\UI\Control;
use Nette\Http\SessionSection;
use Nette\Http\Url;
use Nette\Utils\Validators;
use XPInvest\PayUButton\Client;
use XPInvest\PayUButton\Exceptions\InvalidArgumentException;
use XPInvest\PayUButton\Exceptions\OrderNotFoundException;
use XPInvest\PayUButton\Exceptions\PayUApiException;
use XPInvest\PayUButton\Exceptions\PayUException;
use XPInvest\PayUButton\Order;

class Button extends Control
{

    /** @var callable[] */
    public $onBeforePay = [];

    /** @var callable[] */
    public $onBeforeRedirect = [];

    /** @var callable[] */
    public $onSuccess = [];

    /** @var callable[] */
    public $onError = [];

    /** @var NULL|Order */
    private $order;

    /** @var NULL|string */
    private $notifyUrl;

    /** @var SessionSection */
    private $session;

    /** @var Client */
    private $client;

    /** @var bool */
    private $renderOutput = true;

    /**
     * Button constructor.
     * @param SessionSection $session
     * @param Client $client
     */
    public function __construct(SessionSection $session, Client $client)
    {
        $this->session = $session;
        $this->client = $client;
    }

    /**
     * @param Order $order
     * @return Button
     */
    public function setOrder(Order $order): Button
    {
        $this->order = $order;
        return $this;
    }

    /**
     * @param string $notifyUrl
     * @return Button
     * @throws InvalidArgumentException
     */
    public function setNotifyUrl(string $notifyUrl): Button
    {
        if (!Validators::isUrl($notifyUrl)) {
            throw new InvalidArgumentException('Notify url "' . $notifyUrl . '" is not valid url address.');
        }

        $this->notifyUrl = $notifyUrl;
        return $this;
    }

    /**
     * @return bool
     */
    public function isSandboxActive(): bool
    {
        return $this->client->isSandboxActive();
    }

    public function render()
    {
        if (!$this->renderOutput) {
            return;
        }

        $template = $this->getTemplate();

        if ($template->getFile() === null) {
            $template->setFile(__DIR__ . '/templates/Button.latte');
        }

        $template->render();
    }

    public function handlePay(string $lang = 'cs')
    {
        try {
            $this->onBeforePay();
            $this->validateData();
            $response = $this->client->createOrder(
                $this->order,
                $this->link('//complete!', ['id' => $this->order->getId()]),
                $this->notifyUrl
            );
            $this->session[$this->order->getId()] = $response->getOrderId();
            $this->onBeforeRedirect($response, $this->order);
            $url = (new Url($response->getRedirectUrl()))->setQueryParameter('lang', $lang);
            $this->presenter->redirectUrl($url->getAbsoluteUrl());
        } catch (PayUApiException $e) {
            $this->onError($e);
        }
    }

    public function handleComplete(string $id)
    {
        $this->renderOutput = false;

        if (!isset($this->session[$id])) {
            $this->onError(new OrderNotFoundException);
            return;
        }

        try {
            $payment = $this->client->retrieveOrderPayment($this->session[$id]);
            $this->onSuccess($payment);
        } catch (PayUException $e) {
            $this->onError($e);
        }
    }

    /**
     * @throws InvalidArgumentException
     * @throws OrderNotFoundException
     */
    private function validateData()
    {
        if (!$this->order) {
            throw new OrderNotFoundException;
        } elseif (!$this->notifyUrl) {
            throw new InvalidArgumentException('Missing notify url address.');
        }
    }
}
