<?php declare(strict_types=1);

namespace XPInvest\PayUButton;

use Money\Currency;
use Money\Money;
use Nette\Http\IRequest;
use Nette\Utils\Json;
use OpenPayU_Result;
use stdClass;
use XPInvest\PayUButton\Exceptions\InvalidRequestException;
use XPInvest\PayUButton\Exceptions\PayUApiException;
use XPInvest\PayUButton\Factory\NewOrderArrayFactory;
use XPInvest\PayUButton\Order\Payment;
use XPInvest\PayUButton\Response\NewOrderResponse;

class Client
{

    /** @var bool */
    private $sandbox;

    /** @var string */
    private $clientId;

    /** @var string */
    private $secret;

    /** @var string */
    private $key;

    /** @var \OauthCacheInterface */
    private $oauthCache;

    /**
     * Client constructor.
     * @param bool $sandbox
     * @param string $clientId
     * @param string $secret
     * @param string $key
     * @param \OauthCacheInterface $oauthCache
     */
    public function __construct(
        bool $sandbox,
        string $clientId,
        string $secret,
        string $key,
        \OauthCacheInterface $oauthCache
    ) {
        $this->sandbox = $sandbox;
        $this->clientId = $clientId;
        $this->secret = $secret;
        $this->key = $key;
        $this->oauthCache = $oauthCache;
    }

    /**
     * @return bool
     */
    public function isSandboxActive(): bool
    {
        return $this->sandbox;
    }

    /**
     * @param Order $order
     * @param string $continueUrl
     * @param string $notifyUrl
     * @return NewOrderResponse
     * @throws PayUApiException
     */
    public function createOrder(Order $order, string $continueUrl, string $notifyUrl): NewOrderResponse
    {
        $data = NewOrderArrayFactory::create($order, $this->clientId, $continueUrl, $notifyUrl, $this->sandbox);
        try {
            $this->setupEnvironment();

            /** @var OpenPayU_Result $response */
            $response = \OpenPayU_Order::create($data);
            if ($response->getStatus() === \OpenPayU_Order::SUCCESS) {
                /**
                 * PayU client má úplně rozbitý typehinty ¯\_(ツ)_/¯
                 * @var stdClass $innerResponse
                 */
                $innerResponse = $response->getResponse();

                return new NewOrderResponse(
                    $order->getId(),
                    $innerResponse->orderId,
                    $innerResponse->redirectUri
                );
            }

            throw $this->invalidResponseException($response);
        } catch (\OpenPayU_Exception $e) {
            throw new PayUApiException('Something went wrong during api communication.', 0, $e);
        }
    }

    /**
     * @param string $id
     * @return Order\Payment
     * @throws PayUApiException
     */
    public function retrieveOrderPayment(string $id): Order\Payment
    {
        try {
            $this->setupEnvironment();

            $response = \OpenPayU_Order::retrieve($id);
            if ($response->getStatus() === \OpenPayU_Order::SUCCESS) {
                /**
                 * PayU client má úplně rozbitý typehinty ¯\_(ツ)_/¯
                 * @var stdClass $innerResponse
                 */
                $innerResponse = $response->getResponse();

                return $this->createPaymentDTO($innerResponse->orders[0]);
            }

            throw $this->invalidResponseException($response);
        } catch (\OpenPayU_Exception $e) {
            throw new PayUApiException('Something went wrong during api communication.', 0, $e);
        }
    }

    /**
     * @param IRequest $request
     * @return Order\Payment
     * @throws PayUApiException
     */
    public function consumeOrderPaymentNotificationRequest(IRequest $request): Order\Payment
    {
        if (! $request->isMethod(IRequest::POST)) {
            throw new InvalidRequestException(
                sprintf(
                    'Expected HTTP method %s, but method is %s',
                    IRequest::POST,
                    $request->getMethod(),
                )
            );
        }

        if (empty($request->getRawBody())) {
            throw new InvalidRequestException('Empty request body provided');
        }

        try {
            $this->setupEnvironment();

            $response = \OpenPayU_Order::consumeNotification($request->getRawBody());

            /**
             * PayU client má úplně rozbitý typehinty ¯\_(ツ)_/¯
             * @var stdClass $innerResponse
             */
            $innerResponse = $response->getResponse();

            return $this->createPaymentDTO($innerResponse->order);
        } catch (\OpenPayU_Exception $e) {
            throw new PayUApiException('Something went wrong during api communication.', 0, $e);
        }
    }

    public function sendSuccessResponse()
    {
        header('HTTP/1.1 200 OK');
        exit;
    }

    public function sendErrorResponse()
    {
        header('HTTP/1.1 422 Error');
        exit;
    }

    /**
     * @throws \OpenPayU_Exception_Configuration
     */
    private function setupEnvironment()
    {
        if ($this->sandbox) {
            \OpenPayU_Configuration::setEnvironment('sandbox');
        } else {
            \OpenPayU_Configuration::setEnvironment('secure');
        }

        \OpenPayU_Configuration::setMerchantPosId($this->clientId);
        \OpenPayU_Configuration::setOauthClientId($this->clientId);
        \OpenPayU_Configuration::setOauthClientSecret($this->secret);
        \OpenPayU_Configuration::setSignatureKey($this->key);
        \OpenPayU_Configuration::setOauthTokenCache($this->oauthCache);
    }

    private function createPaymentDTO(stdClass $order) : Payment
    {
        [$currencyCode, $extOrderId] = explode(':', $order->extOrderId, 2);

        return new Payment(
            $extOrderId,
            $order->orderId,
            new Money($order->totalAmount, new Currency($this->sandbox ? $currencyCode : $order->currencyCode)),
            $order->status
        );
    }

    private function invalidResponseException(OpenPayU_Result $result) : PayUApiException
    {
        return new PayUApiException(
            sprintf(
                'Received PayU response with status "%s", error: %s, data: %s',
                $result->getStatus(),
                $result->getError(),
                Json::encode($result->getResponse()),
            )
        );
    }
}
