<?php declare(strict_types=1);

namespace XPInvest\SmsBranaCz\Factories;

use XPInvest\SmsBranaCz\Responses\SmsResponse;
use XPInvest\SmsBranaCz\Exceptions\ParseResponseException;
use XPInvest\SmsBranaCz\SmsClient;

class ResponseFactory
{

	/**
	 * @param string $action
	 * @param \SimpleXMLElement $xml
	 * @return float|SmsResponse
	 * @throws ParseResponseException
	 * @throws InvalidRequestException
	 */
	public static function create(string $action, \SimpleXMLElement $xml)
	{
		if(isset($xml->err) && intval($xml->err) !== 0) {
			throw new InvalidRequestException('SMS Connect error #'.$xml->err);
		}

		if($action === SmsClient::ACTION_SEND_SMS) {
			return self::createSmsResponse($xml);
		} elseif($action === SmsClient::ACTION_CREDIT_INFO) {
			return self::createdCreditResponse($xml);
		} else {
			throw new InvalidRequestException($action.' is not valid action.');
		}
	}

	/**
	 * @param \SimpleXMLElement $xml
	 * @return SmsResponse
	 * @throws ParseResponseException
	 */
	private static function createSmsResponse(\SimpleXMLElement $xml): SmsResponse
	{
		$fields = ['price', 'sms_count', 'credit', 'sms_id'];

		foreach($fields as $field) {
			if(!isset($xml->$field)) {
				throw new ParseResponseException('Missing response field '.$field);
			}
		}

		return new SmsResponse(floatval($xml->price), intval($xml->sms_count), floatval($xml->credit), intval($xml->sms_id));
	}

	private static function createdCreditResponse(\SimpleXMLElement $xml): float
	{
		if(!isset($xml->credit)) {
			throw new ParseResponseException('Missing response field credit');
		}
		return floatval($xml->credit);
	}

}
