<?php declare(strict_types=1);

namespace XPInvest\SmsBranaCz;

use Nette\SmartObject;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use XPInvest\SmsBranaCz\Utils\QueryDateTime;
use XPInvest\SmsBranaCz\Exceptions\InvalidArgumentException;

/**
 * @property-read string $number
 * @property-read string $message
 * @property-read string $dateTimeOfSend
 * @property-read string $deliveryReport
 * @property-read string $senderId
 * @property-read string $senderPhone
 * @property-read string $userId
 * @property-read string $dataCode
 * @property-read string $answerMail
 * @property-read string $deliveryMail
 */
class Sms
{

	use SmartObject;

	const DATA_CODE_7BIT = '7bit';
	const DATA_CODE_UNICODE = 'ucs2';

	/** @var string */
	private $number;

	/** @var string */
	private $message;

	/** @var string */
	private $dateTimeOfSend;

	/** @var string */
	private $deliveryReport;

	/** @var string */
	private $senderId;

	/** @var string */
	private $senderPhone;

	/** @var string */
	private $userId;

	/** @var string */
	private $dataCode;

	/** @var string */
	private $answerMail;

	/** @var string */
	private $deliveryMail;

	/**
	 * Sms constructor.
	 * @param string $number
	 * @param string $message
	 * @param bool $deliveryReport
	 * @throws InvalidArgumentException
	 */
	public function __construct(string $number, string $message, bool $deliveryReport = FALSE)
	{
		$this->number = $number;
		$this->message = Strings::toAscii(Strings::trim($message));
		if(Strings::length($this->message) > 459) {
			throw new InvalidArgumentException('Message string is longer than allowed 459 characters. Message length: '.Strings::length($this->message));
		}
		if($deliveryReport) {
			$this->deliveryReport = '1';
		} else {
			$this->deliveryReport = '0';
		}
	}

	/**
	 * @return string
	 */
	public function getNumber(): string
	{
		return $this->number;
	}

	/**
	 * @return string
	 */
	public function getMessage(): string
	{
		return $this->message;
	}

	/**
	 * @return string
	 */
	public function getDeliveryReport(): string
	{
		return $this->deliveryReport;
	}

	/**
	 * @return string|NULL
	 */
	public function getDateTimeOfSend()
	{
		return $this->dateTimeOfSend;
	}

	/**
	 * @param QueryDateTime $dateTimeOfSend
	 * @return Sms
	 */
	public function withDateTimeOfSend(QueryDateTime $dateTimeOfSend): Sms
	{
		$this->dateTimeOfSend = strval($dateTimeOfSend);
		return $this;
	}

	/**
	 * @return string|NULL
	 */
	public function getSenderId()
	{
		return $this->senderId;
	}

	/**
	 * @param string $senderId
	 * @return Sms
	 */
	public function withSenderId(string $senderId): Sms
	{
		$this->senderId = $senderId;
		return $this;
	}

	/**
	 * @return string|NULL
	 */
	public function getSenderPhone()
	{
		return $this->senderPhone;
	}

	/**
	 * @param string $senderPhone
	 * @return Sms
	 */
	public function withSenderPhone(string $senderPhone): Sms
	{
		$this->senderPhone = $senderPhone;
		return $this;
	}

	/**
	 * @return string|NULL
	 */
	public function getUserId()
	{
		return $this->userId;
	}

	/**
	 * @param string $userId
	 * @return Sms
	 */
	public function withUserId(string $userId): Sms
	{
		$this->userId = $userId;
		return $this;
	}

	/**
	 * @return string|NULL
	 */
	public function getDataCode()
	{
		return $this->dataCode;
	}

	/**
	 * "7bit" - Standard 160 characters SMS (default)
	 * "ucs2" - Unicode message, 70 characters/sms
	 * @param string $dataCode
	 * @return Sms
	 * @throws InvalidArgumentException
	 */
	public function withDataCode(string $dataCode = self::DATA_CODE_7BIT): Sms
	{
		if(!in_array($dataCode, [self::DATA_CODE_7BIT, self::DATA_CODE_7BIT])) {
			throw new InvalidArgumentException('Invalid sms Data Code choosen. It should be one of these: '.self::DATA_CODE_7BIT.' '.self::DATA_CODE_7BIT);
		}
		$this->dataCode = $dataCode;
		return $this;
	}

	/**
	 * @return string|NULL
	 */
	public function getAnswerMail()
	{
		return $this->answerMail;
	}

	/**
	 * @param string $answerMail
	 * @return Sms
	 * @throws InvalidArgumentException
	 */
	public function withAnswerMail(string $answerMail): Sms
	{
		if(!Validators::isEmail($answerMail)) {
			throw new InvalidArgumentException($answerMail.' is not valid e-mail address.');
		}
		$this->answerMail = $answerMail;
		return $this;
	}

	/**
	 * @return string|NULL
	 */
	public function getDeliveryMail()
	{
		return $this->deliveryMail;
	}

	/**
	 * @param string $deliveryMail
	 * @return Sms
	 * @throws InvalidArgumentException
	 */
	public function withDeliveryMail(string $deliveryMail): Sms
	{
		if(!Validators::isEmail($deliveryMail)) {
			throw new InvalidArgumentException($deliveryMail.' is not valid e-mail address.');
		}
		$this->deliveryMail = $deliveryMail;
		return $this;
	}

}
