<?php declare(strict_types=1);

namespace XPInvest\SmsBranaCz;

use GuzzleHttp;
use XPInvest\SmsBranaCz\ISmsClient;
use XPInvest\SmsBranaCz\Sms;
use XPInvest\SmsBranaCz\Responses\SmsResponse;
use XPInvest\SmsBranaCz\Factories\RequestFactory;
use XPInvest\SmsBranaCz\Factories\ResponseFactory;
use XPInvest\SmsBranaCz\Exceptions\SmsClientException;
use XPInvest\SmsBranaCz\Exceptions\InvalidRequestException;
use XPInvest\SmsBranaCz\Exceptions\ParseResponseException;
use XPInvest\SmsBranaCz\Exceptions\ServerException;

class SmsClient implements ISmsClient
{

	const ACTION_SEND_SMS = 'send_sms';
	const ACTION_CREDIT_INFO = 'credit_info';

	/** @var GuzzleHttp\Client */
	private $client;

	/** @var string */
	private $login;

	/** @var string */
	private $password;

	/**
	 * SmsClient constructor.
	 * @param string $login
	 * @param string $password
	 */
	public function __construct(string $login, string $password)
	{
		$this->login = $login;
		$this->password = $password;
		$this->client = new GuzzleHttp\Client([
			'base_uri' => 'https://api.smsbrana.cz/smsconnect/http.php',
			'timeout' => 5.0
		]);
	}

	/**
	 * @param Sms $sms
	 * @return SmsResponse
	 * @throws SmsClientException
	 */
	public function send(Sms $sms): SmsResponse
	{
		return $this->sendRequest(self::ACTION_SEND_SMS, [
			'number' => $sms->number,
			'message' => $sms->message,
			'when' => $sms->dateTimeOfSend,
			'delivery_report' => $sms->deliveryReport,
			'sender_id' => $sms->senderId,
			'sender_phone' => $sms->senderPhone,
			'user_id' => $sms->userId,
			'data_code' => $sms->dataCode,
			'answer_mail' => $sms->answerMail,
			'delivery_mail' => $sms->deliveryMail
		]);
	}

	/**
	 * @return float
	 * @throws SmsClientException
	 */
	public function getCreditInfo(): float
	{
		return $this->sendRequest(self::ACTION_CREDIT_INFO, []);
	}

	/**
	 * @param string $action
	 * @param array $requestParameters
	 * @return SmsResponse|float
	 * @throws InvalidRequestException
	 */
	private function sendRequest(string $action, array $requestParameters)
	{
		$queryParameters = RequestFactory::createQueryParameters($action, $requestParameters, $this->login, $this->password);
		try {
			$response = $this->client->get('', ['query' => $queryParameters]);
		} catch(GuzzleHttp\Exception\ServerException $e) {
			throw new ServerException('Error 500 - No server response. '.$e->getMessage());
		} catch(GuzzleHttp\Exception\ClientException $e) {
			throw new InvalidRequestException('Error '.$e->getCode().' - '.$e->getMessage());
		}

		return ResponseFactory::create($action, self::createXmlFromString(strval($response->getBody())));
	}

	/**
	 * @param string $xmlResponse
	 * @return \SimpleXMLElement
	 * @throws ParseResponseException
	 */
	private static function createXmlFromString(string $xmlResponse): \SimpleXMLElement
	{
		$xml = simplexml_load_string($xmlResponse);

		if(!$xml) {
			throw new ParseResponseException('Cannot create XML object from response');
		}

		return $xml;
	}

}
